#!/usr/bin/env python
#   Copyright 2008 Sammy Fischer
#
#   This file is part of Ice Ice Penguin.
#
#   Ice Ice Penguin is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   Ice Ice Penguin is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with Ice Ice Penguin.  If not, see <http://www.gnu.org/licenses/>.
#
#  You can contact me at sammy.fischer@gmail.com
#
import pygame
from pygame.locals import *
from sys import exit
import random
import math
import os
import time
import glob

IIP_VERSION="1.5.1"

_DEBUG_ = 0

############################
# default controls
############################

#keyboard
#joystick Buttons (joystick:button)
#joystick Axis (joystick:axis:value)
IIP_CNTRL_CPL_LEFT= [str(K_w),      str(K_PAGEUP), "1:4",   "0:4"]
IIP_CNTRL_CPL_RIGHT=[str(K_c),      str(K_PAGEDOWN), "1:5",   "0:5"]
IIP_CNTRL_CPL_FIRE= [str(K_SPACE),  str(K_RETURN), "1:0",   "0:0"]
IIP_CNTRL_FLD_UP=   [str(K_z),      str(K_UP), "1:5:-1","0:5:-1"]
IIP_CNTRL_FLD_DOWN= [str(K_s),      str(K_DOWN), "1:5:1", "0:5:1"]
IIP_CNTRL_FLD_LEFT= [str(K_q),      str(K_LEFT), "1:4:-1","0:4:-1"]
IIP_CNTRL_FLD_RIGHT=[str(K_d),      str(K_RIGHT), "1:4:1", "0:4:1"]

############################
# Ice Ice Penguin Field Types
############################
# not timed
IIP_FT_UNTIMED_BOUNDARY = 5
IIP_FT_BLANK = 104
IIP_FT_START = 100
IIP_FT_TARGET1 = 1021
IIP_FT_TARGET2 = 1022
IIP_FT_TARGET3 = 1023
IIP_FT_TARGET4 = 1024
IIP_FT_TRAP = 103
IIP_FT_WALL = 101
#everything below boundary that is timed
IIP_FT_TIMED_BOUNDARY = 6
IIP_FT_TURNUP = 0
IIP_FT_TURNLEFT = 1
IIP_FT_TURNDOWN = 2
IIP_FT_TURNRIGHT = 3
IIP_FT_SPEEDDOWN = 4
IIP_FT_SPEEDUP = 5

############################
# directions
############################
IIP_UP = 0
IIP_RIGHT = 1
IIP_DOWN = 2
IIP_LEFT = 3

############################
# additional keys for the menu controls
############################
IIP_RETURN = 99
IIP_USE = 4

############################
# menu list predefines
############################
gameLengthList=[ 60000, 120000, 180000, 240000, 300000, 600000]
gameLengthListIndex=2
lifeTimeList=[ 10000, 15000, 20000, 30000]
lifeTimeListIndex=1
tuxSpeedList=[ 50, 100, 200, 300]
tuxSpeedListIndex=2

############################
# game defaults
############################
IIP_LANGUAGE = "en"
IIP_SFXVOLUME = 40.0
IIP_MUSICVOLUME = 90.0
IIP_JOYAXIS_SENSITIVITY = 0.7

IIP_PENGUINMANIA = 0
IIP_BEARMANIA = 1
IIP_SLOWDOWN = 2
IIP_SPEEDUP = 3
IIP_WIPEOUT = 4
IIP_SHUFFLE = 5

############################
# build the level list
############################
IIP_LEVELLIST = []
lvls = glob.glob( "lvls/*.iip")
for l in lvls:
    IIP_LEVELLIST.append( l[5:-4])
IIP_LEVELLIST.append( "RANDOM")
IIP_LEVEL = IIP_LEVELLIST[0]

############################
# load config file
############################
try:
    IIP_HOME = os.environ['HOME']
except:
    try:
        IIP_HOME = os.environ["HOMEDRIVE"]+os.environ["HOMEPATH"] # Windows
    except:
        IIP_HOME = "."
IIP_CONFIGFILE = os.path.join(IIP_HOME, '.iceicepenguin.conf')

try:
    execfile( IIP_CONFIGFILE)
except:
    print "Error reading config. Some configurations might have been reverted to default."

############################
# indexed game defines
############################
IIP_FIELD_LIFETIME = lifeTimeList[ lifeTimeListIndex]   # time till used controls disappear
IIP_TUXSPEED = tuxSpeedList[ tuxSpeedListIndex]         # penguin speed in pixels/sec
IIP_GAMELENGTH = gameLengthList[ gameLengthListIndex]   # amount of millisecond a game lasts

############################
# fixed game defines
############################
SCREENRESX=1024
SCREENRESY=600
IIP_GFXPREFIX=str(SCREENRESX)+"x"+str(SCREENRESY)+"/"
IIP_MENULINESIZE=30
IIP_MENU_OVERFLOW=3
IIP_STANDARDRATEOFSPAWN = 250
IIP_MAXNUMBEROFBEARS = 3
IIP_PROBABILITYOFBEAR = 5
IIP_PROBABILITYOFSPECIAL = 3
IIP_BEARMALUS = 15
############################
# build the language list
############################
IIP_LANGUAGELIST = []
langs = glob.glob( "menu/*")
for l in langs:
    try:
        execfile( l+"/menu.py")
        IIP_LANGUAGELIST.append( l[5:]+":"+IIP_LANGNAME)
    except:
        print "error reading :"+l+"/menu.py"
        pass



############################################################################################
#
#       M E N U S
#
############################################################################################
def drawMenu( menu):
    menuStartY = 40
    menuStartX = 100
    menuScreen.fill((0,0,0,0))
    screen.blit(menuBackground,(0,0))
    startLine=0
    menuBiggestX = 0
    eventScreen.fill( (0,0,0,0))
    for entry in menu:
        menuBiggestX = shadowFont(fontMenu, entry,0, startLine, (255,255,255), 2,menuScreen, menuBiggestX)
        startLine +=30
    eventScreen.fill( (200,200,200,190), (menuStartX-20, menuStartY-20, menuBiggestX+20+menuStartX, startLine+20+menuStartY))
    return menuBiggestX

def humanizeControl( control):
    "returns a human-readable version of a control scheme"
    if control.count(":") == 3:
        (joy,axis,x,y)=control.split(":")
        if int( x) > 0:
            dirStr = "+X"
        elif int( x) < 0:
            dirStr = "-X"
        elif int( y) > 0:
            dirStr = "+Y"
        else:
            dirStr = "-Y"
        return "PAD "+joy+", HAT "+axis+dirStr
    elif control.count(":") == 2:
        (joy,axis,dir)=control.split(":")
        if int( dir) > 0:
            dirStr = "+"
        else:
            dirStr = "-"
        return "JOY "+joy+", AXIS "+axis+dirStr
    elif  control.count(":") == 1:
        (joy,button)=control.split(":")
        return "JOY "+joy+", BUTTON "+button
    else:
        return pygame.key.name( int(control)).upper()

def menuControls():
    "checks for the player controls and returns a player independant code\
    used for the menu controls"
    for event in pygame.event.get():
        if event.type == QUIT:
            exit()
        if event.type == KEYUP:
            if event.key == K_ESCAPE:
                return IIP_RETURN
        iipEvent = ""
        if event.type == KEYDOWN:
            if event.key == K_ESCAPE:
                pass
            else:
                iipEvent = str( event.key)
        if event.type == JOYBUTTONDOWN:
            iipEvent = str(event.joy)+":"+str(event.button)
        if event.type == JOYAXISMOTION:
            jaValue = "0"
            if event.value > IIP_JOYAXIS_SENSITIVITY:
                jaValue = "1"
            elif event.value < -IIP_JOYAXIS_SENSITIVITY:
                jaValue = "-1"
            iipEvent = str(event.joy)+":"+str(event.axis)+":"+jaValue
        if event.type == JOYHATMOTION:
            (x, y)=event.value
            iipEvent = str(event.joy)+":h"+str(event.hat)+":"+str(x)+":"+str(y)
        if iipEvent != "":
            for player in xrange(0,4):
                if iipEvent ==  IIP_CNTRL_CPL_RIGHT[player]:
                    return IIP_RIGHT
                elif iipEvent ==  IIP_CNTRL_CPL_LEFT[player]:
                    return IIP_LEFT
                elif iipEvent ==  IIP_CNTRL_CPL_FIRE[player]:
                    return IIP_USE
                elif iipEvent ==  IIP_CNTRL_FLD_LEFT[player]:
                    return IIP_LEFT
                elif iipEvent ==  IIP_CNTRL_FLD_RIGHT[player]:
                    return IIP_RIGHT
                elif iipEvent ==  IIP_CNTRL_FLD_UP[player]:
                    return IIP_UP
                elif iipEvent ==  IIP_CNTRL_FLD_DOWN[player]:
                    return IIP_DOWN

def levelMenu():
    global IIP_LEVEL
    global IIP_LEVELLIST
    global IIP_OPTIONSMENU
    # find the current level in the list
    cur = 0
    for i in xrange( 0, len( IIP_LEVELLIST)):
        if( IIP_LEVELLIST[i] == IIP_LEVEL):
            cur = i
            break
    cur += 1
    cur %= len( IIP_LEVELLIST)
    IIP_LEVEL = IIP_LEVELLIST[ cur]
    IIP_OPTIONSMENU=[IIP_menuGAMEPLAY, IIP_menuCONTROLS, IIP_menuSOUND, IIP_menuLANGUAGE+" : "+IIP_LANGNAME, IIP_menuLEVEL+" : "+IIP_LEVEL, IIP_menuFULLSCREEN,IIP_menuRETURN]

def languageMenu():
    global IIP_LANGUAGE

    cur = 0
    # find the current language
    for i in xrange( 0, len( IIP_LANGUAGELIST)):
        (fname, lname) = IIP_LANGUAGELIST[i].split( ":")
        if( fname == IIP_LANGUAGE):
            cur = i
            break
    cur += 1
    cur %= len( IIP_LANGUAGELIST)
    (IIP_LANGUAGE, tmp) = IIP_LANGUAGELIST[ cur].split(":")
    # load menu files
    try:
        execfile( "menu/"+IIP_LANGUAGE+"/menu.py")
        return True
    except:
        print "Error loading menu file!"
        return False
    return False

def getControl( control, player, controlName):
    "grabs a key or gamepad control and checks whether it is already used\
    if it's the case, the current controls are swapped with the controls that\
    used to use the new controlset"
    menuScreen.fill( (0,0,0,0))
    shadowFont( fontMenu, IIP_menuKEYMAPTXT_1+controlName, 40, 100, (255,200,0), 2, menuScreen, 0)
    shadowFont( fontMenu, IIP_menuKEYMAPTXT_2, 40, 140, (255,200,0), 2, menuScreen, 0)
    screen.blit(menuBackground,(0,0))
    screen.blit(menuScreen, (0,0))
    pygame.display.flip()
    iipEvent = ""
    while True:
        for event in pygame.event.get():
            if event.type == QUIT:
                exit()
            if event.type == KEYUP:
                if event.key == K_ESCAPE:
                    saveConfig()
                    sfxPing.play()
                    return
            if event.type == KEYDOWN:
                if event.key == K_ESCAPE:
                    pass
                else:
                    iipEvent = str( event.key)
            if event.type == JOYBUTTONDOWN:
                iipEvent = str(event.joy)+":"+str(event.button)
            if event.type == JOYAXISMOTION:
                jaValue = "0"
                if event.value > IIP_JOYAXIS_SENSITIVITY:
                    jaValue = "1"
                elif event.value < -IIP_JOYAXIS_SENSITIVITY:
                    jaValue = "-1"
                iipEvent = str(event.joy)+":"+str(event.axis)+":"+jaValue
            if event.type == JOYHATMOTION:
                (x, y)=event.value
                iipEvent = str(event.joy)+":h"+str(event.hat)+":"+str(x)+":"+str(y)
        if iipEvent != "":
            for p in xrange( 0, 4):
                if IIP_CNTRL_CPL_LEFT[ p] == iipEvent:
                    IIP_CNTRL_CPL_LEFT[ p] = control[ player]
                    break
                elif IIP_CNTRL_CPL_RIGHT[ p] == iipEvent:
                    IIP_CNTRL_CPL_RIGHT[ p] = control[ player]
                    break
                elif IIP_CNTRL_CPL_FIRE[ p] == iipEvent:
                    IIP_CNTRL_CPL_FIRE[ p] = control[ player]
                    break
                elif IIP_CNTRL_FLD_UP[ p] == iipEvent:
                    IIP_CNTRL_FLD_UP[ p] = control[ player]
                    break
                elif IIP_CNTRL_FLD_DOWN[ p] == iipEvent:
                    IIP_CNTRL_FLD_DOWN[ p] = control[ player]
                    break
                elif IIP_CNTRL_FLD_LEFT[ p] == iipEvent:
                    IIP_CNTRL_FLD_LEFT[ p] = control[ player]
                    break
                elif IIP_CNTRL_FLD_RIGHT[ p] == iipEvent:
                    IIP_CNTRL_FLD_RIGHT[ p] = control[ player]
                    break
            control[player] = iipEvent
            break

def createControlMenu( player):
    global IIP_CNTRL_CPL_LEFT
    global IIP_CNTRL_CPL_RIGHT
    global IIP_CNTRL_CPL_FIRE
    global IIP_CNTRL_FLD_UP
    global IIP_CNTRL_FLD_DOWN
    global IIP_CNTRL_FLD_LEFT
    global IIP_CNTRL_FLD_RIGHT

    # humanize the control names
    cplLeft = humanizeControl( IIP_CNTRL_CPL_LEFT[player])
    cplRight = humanizeControl( IIP_CNTRL_CPL_RIGHT[player])
    cntrUse = humanizeControl( IIP_CNTRL_CPL_FIRE[player])
    fldLeft = humanizeControl( IIP_CNTRL_FLD_LEFT[player])
    fldRight = humanizeControl( IIP_CNTRL_FLD_RIGHT[player])
    fldUp = humanizeControl( IIP_CNTRL_FLD_UP[player])
    fldDown = humanizeControl( IIP_CNTRL_FLD_DOWN[player])

    # create the menu
    menu = [ IIP_menuSELECTIONCURSOR+" "+IIP_menuLEFT+" : "+cplLeft, IIP_menuSELECTIONCURSOR+" "+IIP_menuRIGHT+" : "+cplRight, IIP_menuUSE+" : "+cntrUse, IIP_menuFIELDCURSOR+" "+IIP_menuLEFT+" : "+fldLeft, IIP_menuFIELDCURSOR+" "+IIP_menuRIGHT+" : "+fldRight, IIP_menuFIELDCURSOR+" "+IIP_menuUP+" : "+fldUp, IIP_menuFIELDCURSOR+" "+IIP_menuDOWN+" : "+fldDown, IIP_menuRETURN]
    return menu

def controlMenu( player):
    global IIP_CNTRL_CPL_LEFT
    global IIP_CNTRL_CPL_RIGHT
    global IIP_CNTRL_CPL_FIRE
    global IIP_CNTRL_FLD_UP
    global IIP_CNTRL_FLD_DOWN
    global IIP_CNTRL_FLD_LEFT
    global IIP_CNTRL_FLD_RIGHT

    menu = createControlMenu( player)

    maxLines = len( menu)
    menuStartY = 40
    menuStartX = 100
    menuBiggestX = drawMenu( menu)
    currentLine = 0
    menuSelect = pygame.Surface((menuBiggestX+IIP_MENU_OVERFLOW*2,IIP_MENULINESIZE), SRCALPHA, 32)
    menuSelect.fill( (255,0,0,100))
    while True:
        screen.blit(menuBackground,(0,0))
        screen.blit(eventScreen,(0,0))
        screen.blit(menuSelect, (menuStartX-IIP_MENU_OVERFLOW, menuStartY+(currentLine*IIP_MENULINESIZE)+IIP_MENU_OVERFLOW))
        screen.blit(menuScreen, (menuStartX, menuStartY))
        iipEvent = menuControls()

        if iipEvent == IIP_RETURN:
            saveConfig()
            sfxPing.play()
            return
        if iipEvent == IIP_USE:
            if currentLine == maxLines-1:
                sfxPing.play()
                saveConfig()
                return
            elif currentLine == 0:
                cntrl=IIP_CNTRL_CPL_LEFT
                controlName = IIP_menuSELECTIONCURSOR+" "+IIP_menuLEFT
            elif currentLine == 1:
                cntrl=IIP_CNTRL_CPL_RIGHT
                controlName = IIP_menuSELECTIONCURSOR+" "+IIP_menuRIGHT
            elif currentLine == 2:
                cntrl=IIP_CNTRL_CPL_FIRE
                controlName = IIP_menuSELECTIONCURSOR+" "+IIP_menuUSE
            elif currentLine == 3:
                cntrl=IIP_CNTRL_FLD_LEFT
                controlName = IIP_menuFIELDCURSOR+" "+IIP_menuLEFT
            elif currentLine == 4:
                cntrl=IIP_CNTRL_FLD_RIGHT
                controlName = IIP_menuFIELDCURSOR+" "+IIP_menuRIGHT
            elif currentLine == 5:
                cntrl=IIP_CNTRL_FLD_UP
                controlName = IIP_menuFIELDCURSOR+" "+IIP_menuUP
            else:
                cntrl=IIP_CNTRL_FLD_DOWN
                controlName = IIP_menuFIELDCURSOR+" "+IIP_menuDOWN
            sfxPing.play()
            getControl( cntrl, player, controlName)
            menu = createControlMenu( player)
            drawMenu( menu)
            pygame.display.flip()
            time.sleep(1)
            pygame.event.clear()
        if iipEvent == IIP_DOWN:
            currentLine += 1
            if currentLine >= maxLines:
                currentLine = maxLines-1
        if iipEvent == IIP_UP:
            currentLine -= 1
            if currentLine < 0:
                currentLine = 0
        pygame.display.flip()

#IIP_SOUNDMENU=[IIP_menuVOLUMEMUSIC+" : "+str(IIP_MUSICVOLUME), IIP_menuVOLUMESFX+" : "+str(IIP_SFXVOLUME), IIP_menuRETURN]
def soundMenu():
    global IIP_SOUNDMENU
    global IIP_SFXVOLUME
    global IIP_MUSICVOLUME

    playingMusic = True
    pygame.mixer.music.play(0)
    maxLines = len( IIP_SOUNDMENU)
    menuStartY = 40
    menuStartX = 100
    menuBiggestX = drawMenu( IIP_SOUNDMENU)
    currentLine = 0
    menuSelect = pygame.Surface((menuBiggestX+IIP_MENU_OVERFLOW*2,IIP_MENULINESIZE), SRCALPHA, 32)
    menuSelect.fill( (255,0,0,100))
    while True:
        screen.blit(menuBackground,(0,0))
        screen.blit(eventScreen,(0,0))
        screen.blit(menuSelect, (menuStartX-IIP_MENU_OVERFLOW, menuStartY+(currentLine*IIP_MENULINESIZE)+IIP_MENU_OVERFLOW))
        screen.blit(menuScreen, (menuStartX, menuStartY))
        iipEvent = menuControls()
        if iipEvent == IIP_RETURN:
            saveConfig()
            sfxPing.play()
            pygame.mixer.music.stop()
            return
        if iipEvent == IIP_USE:
            if currentLine == maxLines-1:
                saveConfig()
                pygame.mixer.music.stop()
                sfxPing.play()
                return
            elif currentLine == 0:
                sfxPing.play()
                IIP_MUSICVOLUME += 10.0
                IIP_MUSICVOLUME %= 110
                musvol=float(IIP_MUSICVOLUME/100.0)
                pygame.mixer.music.set_volume( musvol)
            elif currentLine == 1:
                sfxPing.play()
                IIP_SFXVOLUME += 10.0
                IIP_SFXVOLUME %= 110
                setAllSfXVolumes()
            IIP_SOUNDMENU=[ IIP_menuVOLUMEMUSIC+" : "+str(int(IIP_MUSICVOLUME)/10), IIP_menuVOLUMESFX+" : "+str(int(IIP_SFXVOLUME)/10), IIP_menuRETURN]
            drawMenu( IIP_SOUNDMENU)
        if iipEvent == IIP_DOWN:
            currentLine += 1
            if currentLine >= maxLines:
                currentLine = maxLines-1
        if iipEvent == IIP_UP:
            currentLine -= 1
            if currentLine < 0:
                currentLine = 0
        if currentLine == 0:
            if playingMusic == False:
                pygame.mixer.music.play(0)
                playingMusic = True
        else:
            pygame.mixer.music.stop()
            playingMusic = False
        pygame.display.flip()

def toggleFullScreen():
    pygame.display.toggle_fullscreen()
    return

#IIP_PLAYERCONTROLMENU=[IIP_menuBLUE+" "+IIP_menuPLAYER, IIP_menuRED+" "+IIP_menuPLAYER, IIP_menuYELLOW+" "+IIP_menuPLAYER, IIP_menuGREEN+" "+IIP_menuPLAYER, IIP_menuRETURN]
def playerControlMenu():
    maxLines = len( IIP_PLAYERCONTROLMENU)
    menuStartY = 40
    menuStartX = 100
    menuBiggestX = drawMenu( IIP_PLAYERCONTROLMENU)
    currentLine = 0
    menuSelect = pygame.Surface((menuBiggestX+IIP_MENU_OVERFLOW*2,IIP_MENULINESIZE), SRCALPHA, 32)
    menuSelect.fill( (255,0,0,100))

    while True:
        screen.blit(menuBackground,(0,0))
        screen.blit(eventScreen,(0,0))
        screen.blit(menuSelect, (menuStartX-IIP_MENU_OVERFLOW, menuStartY+(currentLine*IIP_MENULINESIZE)+IIP_MENU_OVERFLOW))
        screen.blit(menuScreen, (menuStartX, menuStartY))
        iipEvent = menuControls()
        if iipEvent == IIP_RETURN:
            sfxPing.play()
            return
        if iipEvent == IIP_USE:
            if currentLine == maxLines-1:
                sfxPing.play()
                return
            elif currentLine >= 0 and currentLine < 4:
                sfxPing.play()
                controlMenu(currentLine)
                drawMenu( IIP_PLAYERCONTROLMENU)
        if iipEvent == IIP_DOWN:
            currentLine += 1
            if currentLine >= maxLines:
                currentLine = maxLines-1
        if iipEvent == IIP_USE:
            currentLine -= 1
            if currentLine < 0:
                currentLine = 0
        pygame.display.flip()


# IIP_OPTIONSMENU=[IIP_menuGAMEPLAY, IIP_menuCONTROLS, IIP_menuSOUND, IIP_menuLANGUAGE, IIP_menuFULLSCREEN, IIP_menuRETURN]
def optionMenu():
    maxLines = len( IIP_OPTIONSMENU)
    menuStartY = 40
    menuStartX = 100
    menuBiggestX = drawMenu( IIP_OPTIONSMENU)
    currentLine = 0
    menuSelect = pygame.Surface((menuBiggestX+IIP_MENU_OVERFLOW*2,IIP_MENULINESIZE), SRCALPHA, 32)
    menuSelect.fill( (255,0,0,100))
    while True:
        screen.blit(menuBackground,(0,0))
        screen.blit(eventScreen,(0,0))
        screen.blit(menuSelect, (menuStartX-IIP_MENU_OVERFLOW, menuStartY+(currentLine*IIP_MENULINESIZE)+IIP_MENU_OVERFLOW))
        screen.blit(menuScreen, (menuStartX, menuStartY))
        iipEvent = menuControls()
        if iipEvent == IIP_RETURN:
            saveConfig()
            sfxPing.play()
            return
        if iipEvent == IIP_USE:
            if currentLine == maxLines-1:
                saveConfig()
                sfxPing.play()
                return
            elif currentLine == 0:
                sfxPing.play()
                gameplayMenu()
                drawMenu( IIP_OPTIONSMENU)
            elif currentLine == 1:
                sfxPing.play()
                playerControlMenu()
                drawMenu( IIP_OPTIONSMENU)
            elif currentLine == 2:
                sfxPing.play()
                soundMenu()
                drawMenu( IIP_OPTIONSMENU)
            elif currentLine == 3:
                sfxPing.play()
                languageMenu()
                maxLines = len( IIP_OPTIONSMENU)
                menuBiggestX = drawMenu( IIP_OPTIONSMENU)
                del menuSelect
                menuSelect = pygame.Surface((menuBiggestX+IIP_MENU_OVERFLOW*2,IIP_MENULINESIZE), SRCALPHA, 32)
                menuSelect.fill( (255,0,0,100))
            elif currentLine == 4:
                sfxPing.play()
                levelMenu()
                menuBiggestX = drawMenu( IIP_OPTIONSMENU)
                del menuSelect
                menuSelect = pygame.Surface((menuBiggestX+IIP_MENU_OVERFLOW*2,IIP_MENULINESIZE), SRCALPHA, 32)
                menuSelect.fill( (255,0,0,100))
            elif currentLine == 5:
                toggleFullScreen()
                sfxPing.play()
        if iipEvent == IIP_DOWN:
            currentLine += 1
            if currentLine >= maxLines:
                currentLine = maxLines-1
        if iipEvent == IIP_UP:
            currentLine -= 1
            if currentLine < 0:
                currentLine = 0
        pygame.display.flip()

#IIP_GAMEPLAYMENU=[IIP_menuLIFETIME+" : "+str(IIP_FIELD_LIFETIME)+"''", IIP_menuLENGTH+" : "+str(IIP_GAMELENGTH/1000)+"''",  IIP_menuSPEED+" : "+str(IIP_TUXSPEED)+" pixels/sec", IIP_menuRETURN]
def gameplayMenu():
    global IIP_GAMEPLAYMENU
    global IIP_FIELD_LIFETIME
    global IIP_TUXSPEED
    global IIP_GAMELENGTH
    global gameLengthList
    global gameLengthListIndex
    global lifeTimeList
    global lifeTimeListIndex
    global tuxSpeedList
    global tuxSpeedListIndex

    maxLines = len( IIP_GAMEPLAYMENU)
    menuStartY = 40
    menuStartX = 100
    menuBiggestX = drawMenu( IIP_GAMEPLAYMENU)
    currentLine = 0
    menuSelect = pygame.Surface((menuBiggestX+IIP_MENU_OVERFLOW*2,IIP_MENULINESIZE), SRCALPHA, 32)
    menuSelect.fill( (255,0,0,100))
    while True:
        screen.blit(menuBackground,(0,0))
        screen.blit(eventScreen,(0,0))
        screen.blit(menuSelect, (menuStartX-IIP_MENU_OVERFLOW, menuStartY+(currentLine*IIP_MENULINESIZE)+IIP_MENU_OVERFLOW))
        screen.blit(menuScreen, (menuStartX, menuStartY))
        iipEvent = menuControls()
        if iipEvent == IIP_RETURN:
            saveConfig()
            sfxPing.play()
            return
        if iipEvent == IIP_USE:
            if currentLine == maxLines-1:
                saveConfig()
                sfxPing.play()
                return
            elif currentLine == 0:
                sfxPing.play()
                lifeTimeListIndex +=1
                lifeTimeListIndex %= len( lifeTimeList)
                IIP_FIELD_LIFETIME = lifeTimeList[ lifeTimeListIndex]
            elif currentLine == 1:
                sfxPing.play()
                gameLengthListIndex +=1
                gameLengthListIndex %= len( gameLengthList)
                IIP_GAMELENGTH = gameLengthList[ gameLengthListIndex]
            elif currentLine == 2:
                sfxPing.play()
                tuxSpeedListIndex +=1
                tuxSpeedListIndex %= len( tuxSpeedList)
                IIP_TUXSPEED = tuxSpeedList[ tuxSpeedListIndex]
            IIP_GAMEPLAYMENU=[IIP_menuLIFETIME+" : "+str(IIP_FIELD_LIFETIME/1000)+"''", IIP_menuLENGTH+" : "+str(IIP_GAMELENGTH/60000)+"'",  IIP_menuSPEED+" : "+str(IIP_TUXSPEED)+" pixels/sec", IIP_menuRETURN]
            drawMenu( IIP_GAMEPLAYMENU)
        if iipEvent == IIP_DOWN:
            currentLine += 1
            if currentLine >= maxLines:
                currentLine = maxLines-1
        if iipEvent == IIP_UP:
            currentLine -= 1
            if currentLine < 0:
                currentLine = 0
        pygame.display.flip()


# IIP_MAINMENU=[IIP_menuPLAY, IIP_menuOPTIONS, IIP_menuEXIT]
def mainMenu():
    sfxAmbient.play( -1)
    maxLines = len( IIP_MAINMENU)
    menuStartY = 40
    menuStartX = 100
    menuBiggestX = drawMenu( IIP_MAINMENU)
    currentLine = 0
    menuSelect = pygame.Surface((menuBiggestX+IIP_MENU_OVERFLOW*2,IIP_MENULINESIZE), SRCALPHA, 32)
    menuSelect.fill( (255,0,0,100))
    iipEvent = menuControls()
    while True:
        screen.blit(menuBackground,(0,0))
        screen.blit(eventScreen,(0,0))
        screen.blit(menuSelect, (menuStartX-IIP_MENU_OVERFLOW, menuStartY+(currentLine*IIP_MENULINESIZE)+IIP_MENU_OVERFLOW))
        screen.blit(menuScreen, (menuStartX, menuStartY))
        iipEvent = menuControls()
        if iipEvent == IIP_RETURN:
            exit()
        if iipEvent == IIP_USE:
            if currentLine == 0:
                return
            elif currentLine == 1:
                sfxPing.play()
                optionMenu()
                maxLines = len( IIP_MAINMENU)
                menuBiggestX = drawMenu( IIP_MAINMENU)
                del menuSelect
                menuSelect = pygame.Surface((menuBiggestX+IIP_MENU_OVERFLOW*2,IIP_MENULINESIZE), SRCALPHA, 32)
                menuSelect.fill( (255,0,0,100))
            elif currentLine == 2:
                exit()
        if iipEvent == IIP_DOWN:
            currentLine += 1
            if currentLine >= maxLines:
                currentLine = maxLines-1
        if iipEvent == IIP_UP:
            currentLine -= 1
            if currentLine < 0:
                currentLine = 0
        pygame.display.flip()


############################################################################################
#
#       G A M E  F U N C T I O N S  A N D  C L A S S E S
#
############################################################################################

def _DUMP( string, level=1):
    if _DEBUG_ == level:
        print "DBG: "+string

def stereo_pan(x_coord, screen_width):
    left_volume = float(x_coord) / screen_width
    right_volume = 1.0 - left_volume
    return (left_volume, right_volume)

def dumpLevel():
    for i in range(0, levelSideFieldsV):
        line=""
        for s in range(0, levelSideFieldsH):
            line=line+str(levelField[i][s].type)
        _DUMP( line)

def shadowFont( fnt, txt, x, y, col, offset, targetSurface, bigX=0):
    targetSurface.blit(fnt.render(txt, True, (0,0,0)),( x+offset, y+offset))
    targetSurface.blit(fnt.render(txt, True, col),( x, y))
    (sx,sy)=fnt.size(txt)
    if sx > bigX:
        return sx
    else:
        return bigX

def chooseMax( a,b):
    if a>b:
        return a
    else:
        return b

def readNextChar( fh,s,i):
    global targetFields
    global startFields
    c=fh.read(1)
    if c=="\n":
        return readNextChar( fh,s,i)
    else:
        if c=="#":
            return IIP_FT_WALL
        elif c=="S":
            startFields.append( (s,i))
            return IIP_FT_START
        elif c=="1":
            targetFields[0]=(s,i)
            return IIP_FT_TARGET1
        elif c=="2":
            targetFields[1]=(s,i)
            return IIP_FT_TARGET2
        elif c=="3":
            targetFields[2]=(s,i)
            return IIP_FT_TARGET3
        elif c=="4":
            targetFields[3]=(s,i)
            return IIP_FT_TARGET4
        elif c=="." or c==" ":
            return IIP_FT_BLANK
        else:
            return readNextChar( fh,s,i)

def loadLevel( lvl, lvlscreen, lvlcompo):
    "choose between the two different methods to load a level"
    global levelField
    for i in range(0, len( startFields)):
        startFields.pop()
    if len( levelField) > 0:
        loadAndReinitLevel( lvl, lvlscreen, lvlcompo)
    else:
        loadAndInitLevel( lvl, lvlscreen, lvlcompo)

def loadAndInitLevel( lvl, lvlscreen, lvlcompo):
    "loads a lvl from a file and create the level fields"
    global levelField
    fh=open("lvls/"+lvl+".iip","r")
    for i in range(0, levelSideFieldsV):
        tmp=[]
        for s in range(0, levelSideFieldsH):
            sy = i * fieldFullSize
            sx = s * fieldFullSize
            type=readNextChar( fh,s,i)
            if type != IIP_FT_START:
                tmp.append(LevelField(sx, sy, lvlscreen, type))
            else:
                tmp.append(LevelField(sx, sy, lvlcompo, type))
        levelField.append( tmp)
    dumpLevel()

def loadAndReinitLevel( lvl, lvlscreen, lvlcompo):
    "loads a lvl from a file and only fills the level fields"
    global levelField
    fh=open("lvls/"+lvl+".iip","r")
    for i in range(0, levelSideFieldsV):
        for s in range(0, levelSideFieldsH):
            type=readNextChar( fh,s,i)
            if type != IIP_FT_START:
                levelField[i][s].targetSurface=lvlscreen
                levelField[i][s].setType( type,0)
                levelField[i][s].owner=-1
            else:
                levelField[i][s].targetSurface=lvlcompo
                levelField[i][s].setType( type,0)
                levelField[i][s].owner=-1
            levelField[i][s].draw()
    dumpLevel()

def deleteLevel():
    "set all the created level Fields to BLANKS"
    global levelField
    levelScreen.fill( (0,0,0,0))
    levelComposite.fill( (0,0,0,0))
    levelScreen.blit( background, (0,0))
    for i in range(0, levelSideFieldsV-1):
        for s in range(0, levelSideFieldsH-1):
            levelField[i][s].targetSurface = levelScreen
            levelField[i][s].owner = -1
            levelField[i][s].setType( IIP_FT_BLANK, 0)
            levelField[i][s].draw()

def setAllSfXVolumes():
    sfxvol=float( IIP_SFXVOLUME/100)
    global sfxSplash
    global sfxPing
    global sfxApplause
    global sfxAmbient
    global sfxAlert1minute
    global sfxAlerts
    global sfxSuddenDeath
    global sfxRoar
    sfxSplash.set_volume( sfxvol)
    sfxPing.set_volume( sfxvol)
    sfxApplause.set_volume( sfxvol)
    sfxAmbient.set_volume( sfxvol)
    sfxAlert1minute.set_volume( sfxvol)
    sfxAlerts.set_volume( sfxvol)
    sfxSuddenDeath.set_volume( sfxvol)
    sfxRoar.set_volume( sfxvol)

def initJoysticks():
    global joysticks
    for j in range(0,pygame.joystick.get_count()):
        joy = pygame.joystick.Joystick( j)
        print( "Initializing "+joy.get_name()+" ("+str( joy.get_id())+")")
        joy.init()
        joysticks.append( joy)

def saveConfig():
        try:
            fh = open(IIP_CONFIGFILE,'w')
        except:
            commands.getoutput('touch '+IIP_CONFIGFILE)
            fh = open(IIP_CONFIGFILE,'w')
        config="#\n\
# Ice Ice Penguin config file\n\
# These are just python commands which will be automatically executed at runtime\n\n\
# indexed values. The actual value is taken from a list defined in the game\n\
gameLengthListIndex = "+str(gameLengthListIndex)+" # time till used controls disappear\n\
lifeTimeListIndex = "+str(lifeTimeListIndex)+" # penguin speed\n\
tuxSpeedListIndex = "+str(tuxSpeedListIndex)+" # length of a game\n\n\
# actual values\n\
IIP_LANGUAGE = '"+str(IIP_LANGUAGE)+"'\n\
IIP_SFXVOLUME = "+str(IIP_SFXVOLUME)+"\n\
IIP_MUSICVOLUME = "+str(IIP_MUSICVOLUME)+"\n\
IIP_JOYAXIS_SENSITIVITY = "+str(IIP_JOYAXIS_SENSITIVITY)+"\n\
IIP_LEVEL = '"+str(IIP_LEVEL)+"'\n\
#controls :\n\
#keyboard/joystick Buttons (joystick:button)/joystick Axis (joystick:axis:value)\n\
IIP_CNTRL_CPL_LEFT= ['"+IIP_CNTRL_CPL_LEFT[0]+"','"+IIP_CNTRL_CPL_LEFT[1]+"','"+IIP_CNTRL_CPL_LEFT[2]+"','"+IIP_CNTRL_CPL_LEFT[3]+"']\n\
IIP_CNTRL_CPL_RIGHT= ['"+IIP_CNTRL_CPL_RIGHT[0]+"','"+IIP_CNTRL_CPL_RIGHT[1]+"','"+IIP_CNTRL_CPL_RIGHT[2]+"','"+IIP_CNTRL_CPL_RIGHT[3]+"']\n\
IIP_CNTRL_CPL_FIRE= ['"+IIP_CNTRL_CPL_FIRE[0]+"','"+IIP_CNTRL_CPL_FIRE[1]+"','"+IIP_CNTRL_CPL_FIRE[2]+"','"+IIP_CNTRL_CPL_FIRE[3]+"']\n\
IIP_CNTRL_FLD_UP=  ['"+IIP_CNTRL_FLD_UP[0]+"','"+IIP_CNTRL_FLD_UP[1]+"','"+IIP_CNTRL_FLD_UP[2]+"','"+IIP_CNTRL_FLD_UP[3]+"']\n\
IIP_CNTRL_FLD_DOWN= ['"+IIP_CNTRL_FLD_DOWN[0]+"','"+IIP_CNTRL_FLD_DOWN[1]+"','"+IIP_CNTRL_FLD_DOWN[2]+"','"+IIP_CNTRL_FLD_DOWN[3]+"']\n\
IIP_CNTRL_FLD_LEFT= ['"+IIP_CNTRL_FLD_LEFT[0]+"','"+IIP_CNTRL_FLD_LEFT[1]+"','"+IIP_CNTRL_FLD_LEFT[2]+"','"+IIP_CNTRL_FLD_LEFT[3]+"']\n\
IIP_CNTRL_FLD_RIGHT=['"+IIP_CNTRL_FLD_RIGHT[0]+"','"+IIP_CNTRL_FLD_RIGHT[1]+"','"+IIP_CNTRL_FLD_RIGHT[2]+"','"+IIP_CNTRL_FLD_RIGHT[3]+"']\n\
print 'config file loaded'"
        fh.write(config)
        fh.close()

class LevelField():
    "stores each field of the level and takes care of timer functions in conjunction with special field types"
    blink1Max = 3000
    blink1Min = 2500
    blink2Max = 2000
    blink2Min = 1500
    blink3Max = 1000
    blink3Min = 500

    def __init__(self, screenx, screeny, target, t=IIP_FT_BLANK):
        self.screenX = screenx
        self.screenY = screeny
        self.owner = -1
        self.targetSurface = target
        self.setType( t, 0)
        self.draw()
    def wipeout( self):
        "if this field has a player control field, set its timer to 2 sec."
        if self.timer > 0 and self.type < IIP_FT_TIMED_BOUNDARY:
            self.timer = 2000
            if ((( self.timer <= self.blink1Max) and ( self.timer >= self.blink1Min)) or (( self.timer <= self.blink2Max) and ( self.timer >= self.blink2Min)) or (( self.timer <= self.blink3Max) and ( self.timer >= self.blink3Min))) != True:
                self.draw()

    def update(self, time, playerControlPanel, activePlayerControls):
        "decrease timer if running and revert to IIP_FT_BLANK if timer expired"
        if self.timer > 0 and self.type < IIP_FT_TIMED_BOUNDARY:
            self.timer -= time
            if self.timer <= 0:
                playerControlPanel = newControl( self.owner, playerControlPanel)
                activePlayerControls[ self.owner] -= 1
                self.owner = -1
                self.setType( IIP_FT_BLANK, 0)
            if ((( self.timer <= self.blink1Max) and ( self.timer >= self.blink1Min)) or (( self.timer <= self.blink2Max) and ( self.timer >= self.blink2Min)) or (( self.timer <= self.blink3Max) and ( self.timer >= self.blink3Min))) != True:
                self.draw()
        return (playerControlPanel, activePlayerControls)

    def setType(self, type, player, timer = 0):
        "sets a different type for the field and sets the timer"
        self.type = type
        self.owner = player
        if type < IIP_FT_TIMED_BOUNDARY:
            self.timer = timer
        else:
            self.timer = 0

    def draw(self):
        if self.type >1000:
            self.targetSurface.blit( gfxTarget[ self.type-1021], (self.screenX,self.screenY))
        elif self.type >= IIP_FT_TIMED_BOUNDARY:
            self.targetSurface.blit( gfxLevel[ self.type-100+IIP_FT_TIMED_BOUNDARY], (self.screenX,self.screenY))
        else:
            if self.owner != -1:
                self.targetSurface.blit( gfxId[self.owner], (self.screenX,self.screenY))
            self.targetSurface.blit( gfxLevel[ self.type], (self.screenX,self.screenY))

class Penguin():
    "stores and manages penguin instances"

    def __init__(self, x, y, dir, tuxphase,target):
        self.posX=float(levelField[y][x].screenX)
        self.posY=float(levelField[y][x].screenY)
        self.direction=dir
        self.nextFieldX=x
        self.nextFieldY=y-1
        self.fieldScreenX = levelField[self.nextFieldY][self.nextFieldX].screenX
        self.fieldScreenY = levelField[self.nextFieldY][self.nextFieldX].screenY
        self.targetSurface = target
        self.tuxPhase = tuxphase
        self.tuxPhaseTimer = 0
        self.isDead = False
        self.isPenguin = True
        self.isSpecialPenguin = False

    def update( self, time, multi, levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields):
        "only public method being called. Returns true if the penguin was saved."
        traveledDistance = float(multi*IIP_TUXSPEED*(time/1000.0))
        rc = self.move( traveledDistance, levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
        self.fieldScreenX = levelField[self.nextFieldY][self.nextFieldX].screenX
        self.fieldScreenY = levelField[self.nextFieldY][self.nextFieldX].screenY
        self.tuxPhaseTimer += time
        if self.tuxPhaseTimer > 100:
            self.tuxPhase += 1
            self.tuxPhase %= maxTuxPhase
            self.tuxPhaseTimer = 0
        self.draw(levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
        return (rc, scorePlayer)

    def move(self, distance, levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields):
        "move the penguin and check whether it passed the next checkpoint"
        if self.direction == IIP_UP:    # up
            self.posY -= distance
            if math.floor(self.posY) <= self.fieldScreenY:
                self.posY = self.fieldScreenY
                return self.evaluateField( self.fieldScreenY-math.floor(self.posY), levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
            return 0
        elif self.direction == IIP_RIGHT:   # right
            self.posX += distance
            if math.floor(self.posX) >= self.fieldScreenX:
                self.posX = self.fieldScreenX
                return self.evaluateField( math.floor(self.posX)-self.fieldScreenX, levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
            return 0
        elif self.direction == IIP_DOWN:    # down
            self.posY += distance
            if math.floor(self.posY) >= self.fieldScreenY:
                self.posY = self.fieldScreenY
                return self.evaluateField( math.floor(self.posY)-self.fieldScreenY, levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
            return 0
        elif self.direction == IIP_LEFT:    # left
            self.posX -= distance
            if math.floor(self.posX) <= self.fieldScreenX:
                self.posX = self.fieldScreenX
                return self.evaluateField( self.fieldScreenX-math.floor(self.posX), levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
            return 0

    def evaluateField( self, overflowDistance, levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields):
        "checks whether it's a special field and changes the direction accordingly\
        then continues the movement in standard cases"
        if levelField[self.nextFieldY][self.nextFieldX].type == IIP_FT_BLANK:
            self.draw(levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
            self.checkWay(levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
        elif levelField[self.nextFieldY][self.nextFieldX].type == IIP_FT_TURNRIGHT:
            self.direction=IIP_RIGHT
            self.checkWay(levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
        elif levelField[self.nextFieldY][self.nextFieldX].type == IIP_FT_TURNLEFT:
            self.direction=IIP_LEFT
            self.checkWay(levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
        elif levelField[self.nextFieldY][self.nextFieldX].type == IIP_FT_TURNDOWN:
            self.direction=IIP_DOWN
            self.checkWay(levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
        elif levelField[self.nextFieldY][self.nextFieldX].type == IIP_FT_TURNUP:
            self.direction=IIP_UP
            self.checkWay(levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
        elif levelField[self.nextFieldY][self.nextFieldX].type == IIP_FT_TARGET1:
            if( self.isPenguin == True):
                scorePlayer[0] +=1
                return 1
            elif self.isSpecialPenguin == True:
                scorePlayer[0] +=10
                return 3
            else:
                scorePlayer[0] -=IIP_BEARMALUS
                scorePlayer[0]=chooseMax(scorePlayer[0],0)
                sfxRoar.play()
                return 2
        elif levelField[self.nextFieldY][self.nextFieldX].type == IIP_FT_TARGET2:
            if( self.isPenguin == True):
                scorePlayer[1] +=1
                return 1
            elif self.isSpecialPenguin == True:
                scorePlayer[1] +=10
                return 3
            else:
                scorePlayer[1] -=IIP_BEARMALUS
                scorePlayer[1]=chooseMax(scorePlayer[1],0)
                sfxRoar.play()
                return 2
        elif levelField[self.nextFieldY][self.nextFieldX].type == IIP_FT_TARGET3:
            if( self.isPenguin == True):
                scorePlayer[2] +=1
                return 1
            elif self.isSpecialPenguin == True:
                scorePlayer[2] +=10
                return 3
            else:
                scorePlayer[2] -=IIP_BEARMALUS
                scorePlayer[2]=chooseMax(scorePlayer[2],0)
                sfxRoar.play()
                return 2
        elif levelField[self.nextFieldY][self.nextFieldX].type == IIP_FT_TARGET4:
            if( self.isPenguin == True):
                scorePlayer[3] +=1
                return 1
            elif self.isSpecialPenguin == True:
                scorePlayer[3] +=10
                return 3
            else:
                scorePlayer[3] -=IIP_BEARMALUS
                scorePlayer[3]=chooseMax(scorePlayer[3],0)
                sfxRoar.play()
                return 2
        else:
            self.draw(levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
            self.checkWay(levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
        return 0

    def checkWay(self, levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields):
        "checks whether the way in the penguin's direction is free and turns it clockwise until it finds a free way\
        then sets the new destination field"
        if self.direction == 0 and ((self.nextFieldY-1) >= 0 and (levelField[self.nextFieldY-1][self.nextFieldX].type > IIP_FT_WALL or levelField[self.nextFieldY-1][self.nextFieldX].type < 100)):
            self.nextFieldY -=1
        elif self.direction == 1 and ((self.nextFieldX+1) < levelSideFieldsH and (levelField[self.nextFieldY][self.nextFieldX+1].type > IIP_FT_WALL or levelField[self.nextFieldY][self.nextFieldX+1].type < 100)):
            self.nextFieldX +=1
        elif self.direction == 2 and ((self.nextFieldY+1) < levelSideFieldsV and (levelField[self.nextFieldY+1][self.nextFieldX].type > IIP_FT_WALL or levelField[self.nextFieldY+1][self.nextFieldX].type < 100 )):
            self.nextFieldY +=1
        elif self.direction == 3 and ( (self.nextFieldX-1) >= 0 and (levelField[self.nextFieldY][self.nextFieldX-1].type > IIP_FT_WALL or levelField[self.nextFieldY][self.nextFieldX-1].type < 100)):
            self.nextFieldX -=1
        else:
            self.direction +=1
            self.direction %=4
            self.checkWay( levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)

    def inside( self,x,y):
        "only usefull for polarbears. Checks whether the position x,y is inside the killradius of the current bear"
        killRadius = 50
        if ((x<self.posX+killRadius) and (x>self.posX-killRadius)) and ((y<self.posY+killRadius) and (y>self.posY-killRadius)):
            return True
        else:
            return False

    def hunt( self, penguins, deathRow):
        "only usefull for polarbears. Checks all the penguins to see if they are within the hunting boundaries of the\
        bear and if yes, then add them to deathRow"
        if( self.isPenguin == True):
            return
        for p in range( 0, len(penguins)):
            penguin = penguins[ p]
            if self.inside( penguin.posX, penguin.posY) and ((penguin.isPenguin == True) or (penguin.isSpecialPenguin == True)) and (deathRow.count( p) == 0):
                deathRow.append( p)
        return deathRow

    def draw(self, levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields):
        "draws the penguin on the surface defined when the instance was created"
        if self.isPenguin == True:
            self.targetSurface.blit(gfxTux[self.direction][self.tuxPhase], (    math.floor(self.posX),math.floor(self.posY)))
        elif self.isSpecialPenguin == True:
            self.targetSurface.blit(gfxSpecialTux[self.direction][self.tuxPhase], (    math.floor(self.posX),math.floor(self.posY)))
        else:
            self.targetSurface.blit(gfxBear[self.direction][self.tuxPhase], (    math.floor(self.posX)-25,math.floor(self.posY)-25))

def showScores( scorePlayer):
    playerColor=[(0,0,255), (255,0,0), (255,255,0), (0, 255, 0)]
    winner=-1
    winScore = -1
    winColor = (0,0,0)
    loseText = []
    winText = ""
    for p in range( 0,4):
        if scorePlayer[p] > winScore:
            if winner != -1:
                loseText.append( (IIP_menuCOLORS[ winner]+" : "+str(winScore), playerColor[ winner]))
            winner = p
            winScore = scorePlayer[ p]
            winColor = playerColor[ p]
        else:
                tmp=[IIP_menuCOLORS[ p]+" : "+str(scorePlayer[ p]), playerColor[ p]]
                loseText.append( tmp)

    eventScreen.fill((0,0,0, 150))

    winText = IIP_menuCOLORS[ winner]+" "+IIP_endgameWIN
    (x,y)=fontScore.size( winText)
    offX=int(math.floor((SCREENRESX-x)/2))
    shadowFont( fontScore, winText, offX, 150, winColor, 3, eventScreen, 0)
    offY=150+y+10
    winText = IIP_endgameWITH+" "+str(winScore)+" "+IIP_endgamePENGUINS+"!"
    (x,y)=fontScore.size( winText)
    offX=int(math.floor((SCREENRESX-x)/2))
    shadowFont( fontScore, winText, offX, offY, winColor, 3, eventScreen, 0)
    offY=offY+y+10
    for p in range( 0,3):
        (x,y)=fontMenu.size( loseText[p][0])
        offX=int(math.floor((SCREENRESX-x)/2))
        shadowFont( fontMenu, loseText[p][0], offX, offY, loseText[p][1], 2, eventScreen, 0)
        offY=offY+y+5

    screen.blit( eventScreen, (0,0))
    sfxApplause.play()
    pygame.display.flip()
    time.sleep(2)
    while True:
        iipEvent = menuControls()
        if iipEvent == IIP_RETURN or iipEvent == IIP_USE:
            break

def drawControlPanel( playerControlPanel):
    startX = 0
    cplScreen.fill((0,0,0,0))
    for player in range( 0, 4):
        for i in range( 0, 5):
            if playerControlPanel[player][i] != -1:
                cplScreen.blit( gfxLevel[playerControlPanel[player][i]], (startX, 0))
            startX += 50
        startX +=1

def drawControlPanelCursor():
    cplCursorScreen.fill((0,0,0,0))
    for player in range( 0, 4):
        cplCursorScreen.blit( cpl_cursor, (cplStartX[player]+50*cursorCpl[player], 0))

def drawFieldCursor():
    levelCursorScreen.fill((0,0,0,0))
    for player in range( 0, 4):
        levelCursorScreen.blit( levelCursorGfx[player], (cursorFld[player][0]*50, cursorFld[player][1]*50))

def newControl( player, playerControlPanel):
    "search the first empty slot in player's control panel and insert a new control field"
    for i in xrange(0,5):
        if playerControlPanel[player][i] == -1:
            playerControlPanel[player][i] = random.randint(0, 3)
            break
    return playerControlPanel

def eventShuffle( player, playerControlPanel):
    "replace every control the player still has in his control panel"
    for i in xrange(0,5):
        if playerControlPanel[player][i] != -1:
            playerControlPanel[player][i] = random.randint(0, 3)
    return playerControlPanel

def drawScore( scoresPosition, scorePlayer):
    "writes the scores on top of the target fields"
    for ply in range(0,4):
        screen.blit(fontVersion.render(str(scorePlayer[ply]), True, (0,0,0)), scoresPosition[ply])

def isWinner( scorePlayer):
    "checks whether any player has more than its opponents and return True if that's the case"
    maxScore = 0
    unique = False
    for p in range(0,4):
        if scorePlayer[p] > maxScore:
            unique = True
            maxScore = scorePlayer[p]
        elif scorePlayer[p] == maxScore:
            unique = False
    return unique

def centerFont( fnt, txt):
    (x,y)=fnt.size( txt)
    xPos = 512-(x/2)
    return xPos

############################################################################################
#
#       M A I N  G A M E  L O O P
#
############################################################################################

def gameOn( lvl, levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields):
    "setups and runs a game using the level specified"

    activePlayerControls=[0,0,0,0]
    scorePlayer=[0,0,0,0]
    scoresPosition=[(0,0),(0,0),(0,0),(0,0)]
    playerControlPanel=[]
    timesink = 0
    specialTimesink = 0
    numberTimesink = 0
    clockedTime = 0 # incremental clock used for timed event
    cplStartX=[0, 251, 502, 753]
    spawnWaitTime=0
    numberOfPenguins=0
    returnFromGame = False
    IIP_RATEOFSPAWN = IIP_STANDARDRATEOFSPAWN
    ############################
    # restore the current language
    # and create the special events surfaces
    ############################
    try:
        execfile( "menu/"+IIP_LANGUAGE+"/menu.py")
    except:
        pass
    minute1Event = fontScore.render( IIP_event1MINUTE, True, (0,255,0,100))
    minute1XPos = centerFont( fontScore, IIP_event1MINUTE)
    seconds30Event = fontScore.render( "30 "+IIP_eventSECONDS, True, (255,200,0,100))
    seconds30XPos = centerFont( fontScore, "30 "+IIP_eventSECONDS)
    seconds10Event = fontScore.render( "10 "+IIP_eventSECONDS, True, (255,150,0,100))
    seconds10XPos = centerFont( fontScore, "10 "+IIP_eventSECONDS)
    seconds3Event = fontScore.render( "3", True, (255,100,0,100))
    seconds3XPos = centerFont( fontScore, "3")
    seconds2Event = fontScore.render( "2", True, (255,50,0,100))
    seconds2XPos = centerFont( fontScore, "2")
    seconds1Event = fontScore.render( "1", True, (255,0,0,100))
    seconds1XPos = centerFont( fontScore, "1")
    suddenDeathEvent = fontScore.render( IIP_eventSUDDENDEATH, True, (255,0,255,100))
    suddenDeathXPos = centerFont( fontScore, IIP_eventSUDDENDEATH)

    # special event
    penguinManiaEvent = fontScore.render( IIP_eventPENGUINMANIA, True, (0,0,255,100))
    penguinManiaXPos = centerFont( fontScore, IIP_eventPENGUINMANIA)
    bearManiaEvent = fontScore.render( IIP_eventBEARMANIA, True, (255,0,0,100))
    bearManiaXPos = centerFont( fontScore, IIP_eventBEARMANIA)
    slowdownEvent = fontScore.render( IIP_eventSLOWDOWN, True, (0,255,255,100))
    slowdownXPos = centerFont( fontScore, IIP_eventSLOWDOWN)
    speedupEvent = fontScore.render( IIP_eventSPEEDUP, True, ( 100,255,0,100))
    speedupXPos = centerFont( fontScore, IIP_eventSPEEDUP)
    wipeoutEvent = fontScore.render( IIP_eventWIPEOUT, True, ( 0,0,0,100))
    wipeoutXPos = centerFont( fontScore, IIP_eventWIPEOUT)
    shuffleEvent = fontScore.render( IIP_eventSHUFFLE, True, ( 255,255,0,100))
    shuffleXPos = centerFont( fontScore, IIP_eventSHUFFLE)
    specialEventDisplay = None
    specialEventXPos = 0
    specialEventTime = -1
    multiplicator = 1.0
    bearMania = False
    ############################
    # load the level
    ############################
    loadLevel(lvl, levelScreen, levelComposite)
    for ply in range(0,4):
        tmp=[]
        for cpl in range(0, 5):
            tmp.append( random.randint( 0,3))
        playerControlPanel.append( tmp)
        # setup the start positions of the field cursors
        (startX,startY) = targetFields[ply]
        cursorFld[ply]=[int(startX),int(startY)]
        scoresPosition[ply] = ( (int( startX)*50)+25, (int(startY)*50)+25)

    currentStartField = 0
    amountStartField = len( startFields)
    (startX,startY) = startFields[currentStartField]

    drawControlPanel( playerControlPanel)
    drawControlPanelCursor()

    ############################
    # setup the polarbears
    ############################
    numberOfBears = 0
    polarBears = []

    ############################
    # replace the target surface for the level fields
    ############################
    for i in range( 0, levelSideFieldsV):
        for s in range( 0, levelSideFieldsH):
            levelField[i][s].targetSurface = eventScreen

    ############################
    # start the music
    ############################
    if IIP_MUSICVOLUME > 0:
        pygame.mixer.music.play(5)
    else:
        pygame.mixer.music.stop()

    ############################
    # setup the game clock
    ############################
    specialClock = pygame.time.Clock()
    clock = pygame.time.Clock()

    FPS = 0
    curFPS = 0
    framesdrawn = 0
    allFPS = 0

    ############################
    # setup the special event clockframes
    # and the flags showing whether the sound has been played already
    ############################
    minute1Min=IIP_GAMELENGTH-(1000*60)
    minute1Max=IIP_GAMELENGTH-(1000*55)
    f1m = False
    seconds30Min=IIP_GAMELENGTH-(1000*30)
    seconds30Max=IIP_GAMELENGTH-(1000*25)
    f30s = False
    seconds10Min=IIP_GAMELENGTH-(1000*10)
    seconds10Max=IIP_GAMELENGTH-(1000*9)
    f10s = False
    seconds3Min=IIP_GAMELENGTH-(3000)
    seconds3Max=IIP_GAMELENGTH-(2001)
    f3s = False
    seconds2Min=IIP_GAMELENGTH-(2000)
    seconds2Max=IIP_GAMELENGTH-(1001)
    f2s = False
    seconds1Min=IIP_GAMELENGTH-(1000)
    seconds1Max=IIP_GAMELENGTH-(0)
    f1s = False
    suddenDeathMin=IIP_GAMELENGTH
    suddenDeathMax=IIP_GAMELENGTH+(5*1000)
    fSuddenDeath = False

    ############################
    # Loop till the game ends
    ############################
    while True:
        passedTime = clock.tick()
        clockedTime += passedTime
        if passedTime > 0:
            framesdrawn += 1
            curFPS = (1000000/(int(math.floor(passedTime*1000.0))))
            allFPS += curFPS
            FPS = math.floor(float(allFPS/framesdrawn))
        else:
            pass

        if specialEventTime > 0:
            specialEventTime -= passedTime
            if specialEventTime <=8000:
                specialEventDisplay = None
            if specialEventTime < 0:
                IIP_RATEOFSPAWN = IIP_STANDARDRATEOFSPAWN
                multiplicator = 1
                bearMania = False


        ############################
        # Time Events
        ############################
        eventDisplay = None
        eventYPos = 200
        eventXPos = 500
        if (clockedTime >= minute1Min) and (clockedTime < minute1Max):
            eventDisplay = minute1Event
            eventXPos = minute1XPos
            if f1m == False:
                sfxAlert1minute.play()
                f1m = True
        elif (clockedTime >= seconds30Min) and (clockedTime < seconds30Max):
            eventDisplay = seconds30Event
            eventXPos = seconds30XPos
            if f30s == False:
                sfxPing.play()
                f30s = True
        elif (clockedTime >= seconds10Min) and (clockedTime < seconds10Max):
            eventDisplay = seconds10Event
            eventXPos = seconds10XPos
            if f10s == False:
                sfxPing.play()
                f10s = True
        elif (clockedTime >= seconds3Min) and (clockedTime < seconds3Max):
            eventDisplay = seconds3Event
            eventXPos = seconds3XPos
            if f3s == False:
                sfxPing.play()
                f3s = True
        elif (clockedTime >= seconds2Min) and (clockedTime < seconds2Max):
            eventDisplay = seconds2Event
            eventXPos = seconds2XPos
            if f2s == False:
                sfxPing.play()
                f2s = True
        elif (clockedTime >= seconds1Min) and (clockedTime < seconds1Max):
            eventDisplay = seconds1Event
            eventXPos = seconds1XPos
            if f1s == False:
                sfxPing.play()
                f1s = True
        elif (clockedTime >= suddenDeathMin) and (clockedTime < suddenDeathMax):
            eventDisplay = suddenDeathEvent
            eventXPos = suddenDeathXPos
            if fSuddenDeath == False:
                sfxSuddenDeath.play()
                f1s = True

        ############################
        # Time is up
        ############################
        if (clockedTime >= IIP_GAMELENGTH) and (isWinner( scorePlayer) == True):
            eventDisplay = None
            pygame.mixer.music.stop()
            showScores( scorePlayer)
            break

        splashRow=[] # list of saved penguins to delete for each turn
        deathRow=[] # list of dead penguins to delete (different sound)
        screen.blit(levelScreen, (0,0))

        ############################
        # draw the timer bar
        ############################
        timerHeight = int(math.floor(((SCREENRESY-50)*clockedTime)/IIP_GAMELENGTH))
        screen.blit(timerGfx, (1000, timerHeight))

        ############################
        # spawn a new entity and reset the spawn timer
        # from time to time create a bear instead of a penguin
        ############################
        spawnWaitTime += passedTime
        if spawnWaitTime > IIP_RATEOFSPAWN:
            tuxphase = random.randint(0, maxTuxPhase-1)
            tmPenguin = Penguin( startX, startY,0,tuxphase,screen)
            penguins.append( tmPenguin)
            spawnWaitTime = 0
            numberOfPenguins += 1
            currentStartField +=1
            currentStartField %= amountStartField
            (startX, startY) = startFields[ currentStartField]
            if (random.randint(0,100) < IIP_PROBABILITYOFSPECIAL):
                tmPenguin.isPenguin = False
                tmPenguin.isSpecialPenguin = True
            elif ((random.randint(0,100) < IIP_PROBABILITYOFBEAR) and ( numberOfBears < IIP_MAXNUMBEROFBEARS)) or (bearMania == True):
                tmPenguin.isPenguin = False
                polarBears.append( tmPenguin)
                numberOfBears += 1

        ############################
        # update fields
        ############################
        eventScreen.fill((0,0,0,0))
        for i in range( 0, levelSideFieldsV):
            for s in range( 0, levelSideFieldsH):
                (playerControlPanel, activePlayerControls) = levelField[i][s].update( passedTime, playerControlPanel, activePlayerControls)
        screen.blit(eventScreen, (0,0))

        ############################
        # let the bears hunt
        ############################
        for b in polarBears:
            splashRow = b.hunt( penguins, splashRow)

        ############################
        # update the penguins
        ############################
        doSplash = False
        for penguin in range(0, len(penguins)):
            (rc, scorePlayer) = penguins[penguin].update( passedTime,multiplicator, levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
            if rc > 0:
                splashRow.append( penguin)
                if rc == 1:
                    doSplash = True
            if rc == 2:
                polarBears.pop( polarBears.index( penguins[penguin]))
                numberOfBears -= 1
            if rc == 3:
                IIP_RATEOFSPAWN = IIP_STANDARDRATEOFSPAWN
                sfxAlert1minute.play()
                multiplicator = 1
                eventType = random.randint( 0,5)
                bearMania = False
                if eventType == IIP_PENGUINMANIA:
                    specialEventDisplay = penguinManiaEvent
                    specialEventXPos = penguinManiaXPos
                    IIP_RATEOFSPAWN = IIP_STANDARDRATEOFSPAWN / 4
                elif eventType == IIP_BEARMANIA:
                    specialEventDisplay = bearManiaEvent
                    specialEventXPos = bearManiaXPos
                    bearMania = True
                    IIP_RATEOFSPAWN = IIP_STANDARDRATEOFSPAWN * 4
                elif eventType == IIP_SLOWDOWN:
                    multiplicator = 0.5
                    specialEventDisplay = slowdownEvent
                    specialEventXPos = slowdownXPos
                elif eventType == IIP_SPEEDUP:
                    specialEventDisplay = speedupEvent
                    specialEventXPos = speedupXPos
                    multiplicator = 2
                elif eventType == IIP_WIPEOUT:
                    for i in range( 0, levelSideFieldsV):
                        for s in range( 0, levelSideFieldsH):
                            levelField[i][s].wipeout()
                    specialEventDisplay = wipeoutEvent
                    specialEventXPos = wipeoutXPos
                elif eventType == IIP_SHUFFLE:
                    specialEventDisplay = shuffleEvent
                    specialEventXPos = shuffleXPos
                    for p in range(0,4):
                        playerControlPanel = eventShuffle( p, playerControlPanel)
                specialEventTime = 10000
        if doSplash == True:
            sfxSplash.play()

        ############################
        # delete the penguins who were saved or killed
        ############################
        if len( splashRow) > 0:
            splashRow.sort()
            for i in range( len( splashRow), 0, -1):
                del( penguins[ splashRow.pop()])

        for event in pygame.event.get():
            if event.type == QUIT:
                exit()
            if (event.type == MOUSEBUTTONDOWN):
                if( event.type == MOUSEBUTTONDOWN) and (event.button > 2) and (numberTimesink<=1000):
                    pass
                else:
                    pass
            if event.type == KEYUP:
                if event.key == K_ESCAPE:
                    returnFromGame = True
                    break

            iipEvent = ""
            if event.type == KEYDOWN:
                if event.key == K_ESCAPE:
                    pass
                else:
                    iipEvent = str( event.key)
            if event.type == JOYBUTTONDOWN:
                iipEvent = str(event.joy)+":"+str(event.button)

            if event.type == JOYAXISMOTION:
                jaValue = "0"
                if event.value > IIP_JOYAXIS_SENSITIVITY:
                    jaValue = "1"
                elif event.value < -IIP_JOYAXIS_SENSITIVITY:
                    jaValue = "-1"
                iipEvent = str(event.joy)+":"+str(event.axis)+":"+jaValue
            if event.type == JOYHATMOTION:
                (x, y)=event.value
                iipEvent = str(event.joy)+":h"+str(event.hat)+":"+str(x)+":"+str(y)

            if iipEvent != "":
                for player in xrange(0,4):
                    if iipEvent ==  IIP_CNTRL_CPL_RIGHT[player]:
                        cursorCpl[player] +=1
                        if cursorCpl[player] > 4:
                            cursorCpl[player] = 4
                    elif iipEvent ==  IIP_CNTRL_CPL_LEFT[player]:
                        cursorCpl[player] -=1
                        if cursorCpl[player] < 0:
                            cursorCpl[player] = 0
                    elif iipEvent ==  IIP_CNTRL_CPL_FIRE[player]:
                        if (playerControlPanel[player][cursorCpl[player]] != -1) and (levelField[cursorFld[player][1]][cursorFld[player][0]].type == IIP_FT_BLANK) and (activePlayerControls[ player] < 3):
                            levelField[cursorFld[player][1]][cursorFld[player][0]].setType( playerControlPanel[player][cursorCpl[player]], player, IIP_FIELD_LIFETIME)
                            playerControlPanel[player][cursorCpl[player]]=-1
                            activePlayerControls[ player] += 1
                        else:
                            # play some error sound
                            pass
                    elif iipEvent ==  IIP_CNTRL_FLD_LEFT[player]:
                        cursorFld[player][0] -=1
                        if cursorFld[player][0] < 0:
                            cursorFld[player][0] = 0
                    elif iipEvent ==  IIP_CNTRL_FLD_RIGHT[player]:
                        cursorFld[player][0] +=1
                        if cursorFld[player][0] >= levelSideFieldsH:
                            cursorFld[player][0] = levelSideFieldsH-1
                    elif iipEvent ==  IIP_CNTRL_FLD_UP[player]:
                        cursorFld[player][1] -=1
                        if cursorFld[player][1] < 0:
                            cursorFld[player][1] = 0
                    elif iipEvent ==  IIP_CNTRL_FLD_DOWN[player]:
                        cursorFld[player][1] +=1
                        if cursorFld[player][1] >= levelSideFieldsV:
                            cursorFld[player][1] = levelSideFieldsV-1
        if returnFromGame == True:
            break
        drawControlPanel( playerControlPanel)
        drawControlPanelCursor()
        drawFieldCursor()
        screen.blit(levelComposite, (0,0))
        screen.blit(cplScreen, (0,550))
        screen.blit(cplCursorScreen, (0,550))
        screen.blit(levelCursorScreen, (0,0))
        drawScore( scoresPosition, scorePlayer)
        if eventDisplay != None:
            screen.blit( eventDisplay, (eventXPos,eventYPos))
        if specialEventDisplay != None:
            screen.blit( specialEventDisplay, (specialEventXPos,eventYPos+60))
        pygame.display.flip()
    pygame.event.clear()
    pygame.mixer.music.stop()
    print "overall Frames Per Seconds : "+str(FPS)
    return

############################################################################################
#
#       I N I T
#
############################################################################################

pygame.mixer.pre_init(48000)
pygame.init()
pygame.key.set_repeat( 100, 100)

############################
# font init
############################
fontMenu = pygame.font.Font("fonts/chubby.ttf",30)
fontVersion = pygame.font.Font("fonts/chubby.ttf",11)
fontScore = pygame.font.Font("fonts/chubby.ttf",60)

############################
# load samples and music
############################
sfxAmbient=pygame.mixer.Sound("sfx/ambient.ogg")
sfxSplash=pygame.mixer.Sound("sfx/splash.ogg")
sfxPing=pygame.mixer.Sound("sfx/ping.ogg")
sfxApplause=pygame.mixer.Sound("sfx/applause.ogg")
sfxAlert1minute=pygame.mixer.Sound("sfx/alert1minute.ogg")
sfxAlerts=pygame.mixer.Sound("sfx/alerts.ogg")
sfxSuddenDeath=pygame.mixer.Sound("sfx/suddendeath.ogg")
sfxRoar=pygame.mixer.Sound("sfx/roar.ogg")
setAllSfXVolumes()
pygame.mixer.music.load( "sfx/music/retrorific.ogg")
pygame.mixer.music.set_volume( IIP_MUSICVOLUME/100.0)

############################
# initialize the joysticks
############################
joysticks=[]
initJoysticks()

############################
# setup screens and surfaces
############################
screen = pygame.display.set_mode((SCREENRESX,SCREENRESY), SRCALPHA|DOUBLEBUF,32)
menuScreen = pygame.Surface((SCREENRESX,SCREENRESY), SRCALPHA, 32)
levelScreen = pygame.Surface((SCREENRESX,SCREENRESY), SRCALPHA, 32)
levelComposite = pygame.Surface((SCREENRESX,SCREENRESY), SRCALPHA, 32)
levelCursorScreen =  pygame.Surface((SCREENRESX,SCREENRESY-50), SRCALPHA, 32)
eventScreen = pygame.Surface((SCREENRESX,SCREENRESY-50), SRCALPHA, 32)
cplScreen = pygame.Surface((SCREENRESX,50), SRCALPHA, 32)
cplCursorScreen = pygame.Surface((SCREENRESX,50), SRCALPHA, 32)
hudScreen = pygame.Surface((SCREENRESX,SCREENRESY), SRCALPHA, 32)

############################
# Load the images
############################
background = pygame.image.load('gfx/'+IIP_GFXPREFIX+'background.jpg').convert()
menuBackground = pygame.image.load('gfx/'+IIP_GFXPREFIX+'title.jpg').convert()
cpl_cursor = pygame.image.load('gfx/'+IIP_GFXPREFIX+'cplcursor.png').convert_alpha()
timerGfx = pygame.image.load('gfx/'+IIP_GFXPREFIX+'timerbar.png')
levelScreen.blit( background, (0,0))
pygame.display.set_caption("Ice Ice Pengui!")

############################
#add the version number to the background
############################
menuBackground.blit( fontVersion.render("VERSION : "+IIP_VERSION, True, (0,0,0)),( 20, 585))

###########################
# load tux and bears
###########################
gfxTux=[]
gfxSpecialTux=[]
gfxBear=[]
maxTuxPhase=8
tuxPhase = 0
tuxTimesink = 0
for dir in range( 0, 4):
    tmpTux=[]
    tmpSTux=[]
    tmpBear=[]
    for i in range( 0, maxTuxPhase):
        fname="gfx/"+IIP_GFXPREFIX+"tux/"+str(dir)+"/"+str(i)+".png"
        tmpTux.append(pygame.image.load(fname).convert_alpha())
        fname="gfx/"+IIP_GFXPREFIX+"specialtux/"+str(dir)+"/"+str(i)+".png"
        tmpSTux.append(pygame.image.load(fname).convert_alpha())
        fname="gfx/"+IIP_GFXPREFIX+"bear/"+str(dir)+"/"+str(i)+".png"
        tmpBear.append(pygame.image.load(fname).convert_alpha())
    gfxSpecialTux.append(tmpSTux)
    gfxTux.append(tmpTux)
    gfxBear.append(tmpBear)

###########################
# load level gfx
###########################
gfxLevel=[]
for i in range( 0, IIP_FT_TIMED_BOUNDARY):
    fname="gfx/"+IIP_GFXPREFIX+"fields/"+str(i)+".png"
    gfxLevel.append(pygame.image.load(fname).convert_alpha())
for i in range( 0, IIP_FT_UNTIMED_BOUNDARY):
    fname="gfx/"+IIP_GFXPREFIX+"fields/"+str(i+100)+".png"
    gfxLevel.append(pygame.image.load(fname).convert_alpha())
gfxTarget=[]
for i in range( 1, 5):
    fname="gfx/"+IIP_GFXPREFIX+"fields/102"+str(i)+".png"
    gfxTarget.append(pygame.image.load(fname).convert_alpha())

###########################
# load gamecursor gfx
###########################
levelCursorGfx=[]
gfxId=[]
for i in range( 0, 4):
    levelCursorGfx.append(pygame.image.load('gfx/'+IIP_GFXPREFIX+'lvlcursor'+str(i)+'.png').convert_alpha())
    gfxId.append(pygame.image.load('gfx/'+IIP_GFXPREFIX+'id'+str(i)+'.png').convert_alpha())


############################
# default variables
############################
#assume 50 as size
tuxHalfSize = 25
fieldFullSize = 50
levelSideFieldsH = int(math.floor( SCREENRESX/fieldFullSize))
levelSideFieldsV = int(math.floor( SCREENRESY/fieldFullSize)-1)

# load menu files
execfile( "menu/"+IIP_LANGUAGE+"/menu.py")

targetFields=[(),(),(),()]
scorePlayer=[0,0,0,0]
startFields=[]
levelField=[]
penguins=[]
playerControlPanel=[]
activePlayerControls=[]
fieldEvent = [] # stores the event fields used
cursorFld=[[9,4],[11,6],[9,6],[11,4]]
timesink = 0
specialTimesink = 0
numberTimesink = 0
clockedTime = 0 # incremental clock used for timed event
cursorCpl=[0,0,0,0]
cplStartX=[0, 251, 502, 753]
spawnWaitTime=0
numberOfPenguins=0

while True:
    mainMenu()
    pygame.event.clear()
    sfxAmbient.stop()
    lvl="Side 5"
    if IIP_LEVEL == "RANDOM":
        lvl=IIP_LEVELLIST[random.randint(0, len( IIP_LEVELLIST)-2)]
    else:
        lvl=IIP_LEVEL
    gameOn( lvl, levelField, penguins, playerControlPanel, scorePlayer, cursorFld, activePlayerControls, startFields)
    del penguins
    penguins=[]
    deleteLevel()
    sfxAmbient.play(-1)

